/*
 * Decompiled with CFR 0.152.
 */
package moe.plushie.armourers_workshop.core.data;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;
import moe.plushie.armourers_workshop.api.common.IResultHandler;
import moe.plushie.armourers_workshop.core.data.ticket.Ticket;
import moe.plushie.armourers_workshop.utils.ThreadUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;

public class DataTransformer<K, V, T> {
    private final ExecutorService mainThread;
    private final ExecutorService transformThread;
    private final LoadHandler<K, T> loader;
    private final TransformHandler<K, T, V> transformer;
    private final Validator validator = new Validator();
    private final LinkedList<Entry> loadQueue = new LinkedList();
    private final LinkedList<Entry> transformQueue = new LinkedList();
    private final ConcurrentHashMap<K, Entry> allEntries = new ConcurrentHashMap();
    private final AtomicInteger loading = new AtomicInteger(0);
    private final AtomicInteger transforming = new AtomicInteger(0);
    private final int maxLoadCount;
    private final int maxTransformCount;

    public DataTransformer(ThreadFactory config, LoadHandler<K, T> loader, TransformHandler<K, T, V> transformer, int maxLoadCount, int maxTransformCount) {
        this.mainThread = ThreadUtils.newFixedThreadPool(1, config);
        this.transformThread = ThreadUtils.newFixedThreadPool(maxTransformCount, config);
        this.loader = loader;
        this.transformer = transformer;
        this.maxLoadCount = maxLoadCount;
        this.maxTransformCount = maxTransformCount;
    }

    public void remove(K key) {
        this.allEntries.remove(key);
    }

    @Nullable
    public Pair<V, Exception> get(K key) {
        Entry entry = this.getEntry(key);
        if (entry != null) {
            return entry.transformedData;
        }
        return null;
    }

    @Nullable
    public Pair<V, Exception> getOrLoad(K key, Ticket ticket) {
        Entry entry = this.getEntryAndCreate(key);
        if (!entry.isCompleted()) {
            this.load(key, ticket, null);
        }
        return entry.transformedData;
    }

    public void load(K key, Ticket ticket, IResultHandler<V> resultHandler) {
        Entry entry = this.getEntryAndCreate(key);
        this.validator.update(key, ticket);
        entry.notify(resultHandler);
        if (entry.isCompleted()) {
            return;
        }
        entry.elevate(ticket.priority(key));
        this.mainThread.execute(() -> {
            if (!entry.isLoading) {
                entry.isLoading = true;
                this.loadQueue.add(entry);
            }
            this.dispatchIfNeeded();
        });
    }

    public void shutdown() {
        this.loadQueue.clear();
        this.transformQueue.clear();
        this.allEntries.clear();
        this.mainThread.shutdown();
        this.transformThread.shutdown();
        this.loading.set(0);
        this.transforming.set(0);
    }

    private void dispatchIfNeeded() {
        this.doLoadIfNeeded();
        this.doTransformIfNeeded();
    }

    private void doLoadIfNeeded() {
        Entry entry;
        if (this.loading.get() >= this.maxLoadCount) {
            return;
        }
        while ((entry = this.getLastTask(this.loadQueue)) != null) {
            if (entry.isCompleted()) continue;
            if (!this.validator.test(entry.key)) {
                this.abort(entry);
                continue;
            }
            this.load(entry);
            break;
        }
    }

    private void doTransformIfNeeded() {
        Entry entry;
        if (this.transforming.get() >= this.maxTransformCount) {
            return;
        }
        while ((entry = this.getLastTask(this.transformQueue)) != null) {
            if (entry.isCompleted()) continue;
            Object value = entry.getLoadedValue();
            if (value == null || !this.validator.test(entry.key)) {
                this.abort(entry);
                continue;
            }
            this.transform(value, entry);
            break;
        }
    }

    private void load(Entry entry) {
        this.loading.incrementAndGet();
        this.loader.accept(entry.key, (result, exception) -> this.mainThread.execute(() -> {
            this.loading.decrementAndGet();
            entry.receiveLoadResult(result, exception);
            if (!entry.isTransforming) {
                entry.isTransforming = true;
                this.transformQueue.add(entry);
            }
            this.dispatchIfNeeded();
        }));
    }

    private void transform(T value, Entry entry) {
        this.transforming.incrementAndGet();
        this.transformThread.execute(() -> this.transformer.accept(entry.key, value, (result, exception) -> this.mainThread.execute(() -> {
            this.transforming.decrementAndGet();
            entry.receiveTransformResult(result, exception);
            this.dispatchIfNeeded();
        })));
    }

    private void abort(Entry entry) {
        entry.abort();
        this.allEntries.remove(entry.key);
    }

    private Entry getEntry(K key) {
        return this.allEntries.get(key);
    }

    private Entry getEntryAndCreate(K key) {
        return this.allEntries.computeIfAbsent(key, x$0 -> new Entry(x$0));
    }

    @Nullable
    private Entry getLastTask(List<Entry> queue) {
        if (queue.isEmpty()) {
            return null;
        }
        Entry lastEntry = null;
        for (Entry entry : queue) {
            if (lastEntry != null && !(entry.priority > lastEntry.priority)) continue;
            lastEntry = entry;
        }
        queue.remove(lastEntry);
        return lastEntry;
    }

    public class Validator
    implements Predicate<K> {
        private final LinkedList<WeakReference<Ticket>> tickets = new LinkedList();

        @Override
        public synchronized boolean test(K key) {
            for (WeakReference weakReference : this.tickets) {
                Ticket oldTicket = (Ticket)weakReference.get();
                if (oldTicket == null || !oldTicket.contains(key)) continue;
                return true;
            }
            return false;
        }

        public synchronized void update(K key, Ticket ticket) {
            ticket.add(key);
            this.addTicket(ticket);
        }

        private void addTicket(Ticket ticket) {
            Iterator iterator = this.tickets.iterator();
            while (iterator.hasNext()) {
                Ticket oldTicket = (Ticket)((WeakReference)iterator.next()).get();
                if (oldTicket == ticket) {
                    return;
                }
                if (oldTicket != null) continue;
                iterator.remove();
            }
            this.tickets.add(new WeakReference<Ticket>(ticket));
        }
    }

    public static interface LoadHandler<T1, T2> {
        public void accept(T1 var1, IResultHandler<T2> var2);
    }

    public static interface TransformHandler<T1, T2, T3> {
        public void accept(T1 var1, T2 var2, IResultHandler<T3> var3);
    }

    protected class Entry {
        private final K key;
        private ArrayList<IResultHandler<V>> callbacks;
        private Pair<T, Exception> loadedData;
        private Pair<V, Exception> transformedData;
        private float priority = 0.0f;
        private boolean isLoading = false;
        private boolean isTransforming = false;

        Entry(K key) {
            this.key = key;
        }

        public void elevate(float priority) {
            if (this.priority < priority) {
                this.priority = priority;
            }
        }

        public void notify(IResultHandler<V> callback) {
            if (callback == null) {
                return;
            }
            if (this.transformedData != null) {
                callback.apply(this.transformedData.getKey(), (Exception)this.transformedData.getValue());
                return;
            }
            if (this.callbacks == null) {
                this.callbacks = new ArrayList();
            }
            this.callbacks.add(callback);
        }

        public void receiveLoadResult(T value, Exception exception) {
            this.loadedData = Pair.of(value, (Object)exception);
            this.isLoading = false;
        }

        public void receiveTransformResult(V value, Exception exception) {
            this.transformedData = Pair.of(value, (Object)exception);
            this.isTransforming = false;
            this.sendNotify();
        }

        public void abort() {
            this.transformedData = Pair.of(null, (Object)new RuntimeException("abort"));
            this.sendNotify();
        }

        public void sendNotify() {
            ArrayList callbacks = this.callbacks;
            this.callbacks = null;
            if (callbacks == null || callbacks.isEmpty()) {
                return;
            }
            for (IResultHandler callback : callbacks) {
                callback.apply(this.transformedData.getKey(), (Exception)this.transformedData.getValue());
            }
        }

        @Nullable
        public T getLoadedValue() {
            if (this.loadedData != null) {
                return this.loadedData.getKey();
            }
            return null;
        }

        public boolean isCompleted() {
            return this.transformedData != null;
        }
    }

    public static class Builder<K, V, T> {
        private int maxLoadCount = 4;
        private int maxTransformCount = 4;
        private ThreadFactory configure;
        private LoadHandler<K, T> loader;
        private TransformHandler<K, T, V> transformer;

        public Builder<K, V, T> thread(String name, int newPriority) {
            this.configure = r -> {
                Thread thread = new Thread(r, name);
                thread.setPriority(newPriority);
                return thread;
            };
            return this;
        }

        public Builder<K, V, T> loadCount(int count) {
            this.maxLoadCount = count;
            return this;
        }

        public Builder<K, V, T> transformCount(int count) {
            this.maxTransformCount = count;
            return this;
        }

        public Builder<K, V, T> loader(LoadHandler<K, T> handler) {
            this.loader = handler;
            return this;
        }

        public Builder<K, V, T> transformer(TransformHandler<K, T, V> handler) {
            this.transformer = handler;
            return this;
        }

        public DataTransformer<K, V, T> build() {
            return new DataTransformer<K, V, T>(this.configure, this.loader, this.transformer, this.maxLoadCount, this.maxTransformCount);
        }
    }
}

